# Copyright 2021-2025 dilemma-vr.games
# Shared under MIT license

import os
import time
import argparse
import logging
import requests
import RPi.GPIO as GPIO

DILEMMA_API = "https://dilemma-vr.games/api/devices/"

logging.basicConfig(
    format='%(asctime)s %(message)s',
    level=logging.INFO,
    datefmt='%Y-%m-%d %H:%M:%S'
)

headers = {
    "User-Agent": "wardenpi-1.1.0",
}


def log(message):
    logging.info(message)

def poll_api(device_id, open_cb, close_cb):
    """Continously poll API and drive call callbacks"""
    # Resolve API URL
    api_url = os.environ.get("DILEMMA_API", DILEMMA_API)

    i = -1  # Use to throttle API requests
    N = 100 # Idle cycles (each 0.05s)

    # Store game ID once the game reached 'locked' state
    locked_game_id = None

    sess = requests.Session()

    # Exit loop and finish on request
    while True:
        i += 1
        if i % N != 0:
            time.sleep(0.05)
            continue

        try:
            # The API always returns the latest game started by the device
            # owner. To prevent players from starting a new game in case
            # of defeat, DilemmaWarden lockes the API to a specific game
            # ID once the game reaches locked state.
            param = ("?gameid=" + locked_game_id) if locked_game_id else ""
            resp = sess.get(api_url + device_id + param, headers=headers)
        except requests.exceptions.ConnectionError as e:
            log(e)
            continue
            
        if not resp.ok:
            log("Unexpected return code: %d" % resp.status_code)
            continue

        state = resp.text
        log(state)

        if state == "locked":
            close_cb()
        else:
            open_cb()

        header = "X-Game-ID"
        if header in resp.headers:
            if state == "locked" and locked_game_id is None:
                # Lock DilemmaWarden to specific game ID
                locked_game_id = resp.headers[header]
                log("Game ID locked: %s" % locked_game_id)

def main(args):
    """Main interface"""

    def open_cb():
        """Callback for state 'open'"""
        if args.inverse:
            GPIO.output(args.pin, GPIO.HIGH)
        else:
            GPIO.output(args.pin, GPIO.LOW)

    def close_cb():
        """Callback for state 'closed'"""
        if args.inverse:
            GPIO.output(args.pin, GPIO.LOW)
        else:
            GPIO.output(args.pin, GPIO.HIGH)

    try:
        GPIO.setmode(GPIO.BCM)
        GPIO.setup(args.pin, GPIO.OUT)
        poll_api(args.device_id, open_cb, close_cb)
    except KeyboardInterrupt:
        pass
    finally:
        GPIO.cleanup()


if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("device_id", metavar="DEVICE_ID",
                        help="Device ID used as API endpoint")
    parser.add_argument("-p", "--pin", type=int, required=True,
                        help="Raspberry PI pin id controlled by Dilemma VR")
    parser.add_argument("-i", "--inverse",
                        help="Drive pin HIGH if cell is open")

    args = parser.parse_args()
    main(args)
