// Copyright 2021-2025 dilemma-vr.games
// Shared under MIT license

// Version: 1.1.0

#include <SPI.h>
#include <Ethernet.h>

////////////////////////////////////////////////////////////////
// Hardware setup (please check)
const char DEVICE_ID[] = "ENTER_YOUR_DEVICE_ID_HERE";
const int PIN = 8;  // This pin in controlled by the Dilemma VR session
const bool LOCK_VALUE = HIGH;  // State of the pin if the door is locked
const bool FREE_VALUE = LOW;  // State of the pin if the door is open

const int ETHERNET_PIN = 10;
// 10  Most Arduino shields
//  5  MKR ETH shield
//  0  Teensy 2.0
// 20  Teensy++ 2.0
// 15  ESP8266 with Adafruit Featherwing Ethernet
// 33  ESP32 with Adafruit Featherwing Ethernet

////////////////////////////////////////////////////////////////
// Callback configuration (implement custom logic if needed)

void setupPins() {
  pinMode(PIN, OUTPUT);
}

// Callback for started state
void started_cb() {
  digitalWrite(PIN, FREE_VALUE);
}

// Callback for locked state
void locked_cb() {
  digitalWrite(PIN, LOCK_VALUE);
}

// Callback for released state
void released_cb() {
  digitalWrite(PIN, FREE_VALUE);
}

// Callback in case of error
void error_cb() {
  // do nothing
}

////////////////////////////////////////////////////////////////
// Network setup (should be fine)

// Assign a MAC address for the ethernet controller.
byte mac[] = {0x4e, 0xa2, 0xa5, 0x78, 0x1b, 0x6b};

// Host name
char server[] = "dilemma-vr.games";

// Set the static IP address to use if the DHCP fails to assign
IPAddress ip(192, 168, 0, 177);
IPAddress myDns(192, 168, 0, 1);

// initialize the library instance:
EthernetClient client;

////////////////////////////////////////////////////////////////
// Main program
void setup() {
  setupPins();

  Ethernet.init(ETHERNET_PIN);

  // Start serial port for debugging
  Serial.begin(9600);
  while (!Serial) {
    ; // wait for serial port to connect. Needed for native USB port only
  }

  // start the Ethernet connection:
  Serial.println("Initialize Ethernet with DHCP:");
  if (Ethernet.begin(mac) == 0) {
    Serial.println("Failed to configure Ethernet using DHCP");
    // Check for Ethernet hardware present
    if (Ethernet.hardwareStatus() == EthernetNoHardware) {
      Serial.println("Ethernet shield was not found.  Sorry, can't run without hardware. :(");
      error_cb();
      while (true) {
        delay(1); // do nothing, no point running without Ethernet hardware
      }
    }
    if (Ethernet.linkStatus() == LinkOFF) {
      Serial.println("Ethernet cable is not connected.");
    }
    // try to congifure using IP address instead of DHCP:
    Ethernet.begin(mac, ip, myDns);
    Serial.print("My IP address: ");
    Serial.println(Ethernet.localIP());
  } else {
    Serial.print("DHCP assigned IP ");
    Serial.println(Ethernet.localIP());
  }
  // give the Ethernet shield a second to initialize:
  delay(1000);
}


bool game_locked = false;
String locked_game_id;

void loop() {
  // close any connection before send a new request.
  // This will free the socket on the WiFi shield
  client.stop();

  while (true) {
    // if there's a successful connection:
    if (client.connect(server, 80)) {
      // Build and send the HTTP GET request:
      String request("GET /api/devices/");
      request += DEVICE_ID;
      if (game_locked) {
        // Once the game entered locked state, we want to lock the device to
        // that session.
        request += "?gameid=";
        request += locked_game_id;
      }
      request += " HTTP/1.1\r\n";
      request += "Host: dilemma-vr.games\r\n";
      request += "User-Agent: wardenino-1.1.0\r\n";
      request += "Connection: close\r\n\r\n";
      client.print(request);

      // Parse response line
      String resp = client.readStringUntil('\n');
      if (!resp.startsWith("HTTP/1.1 200 ")) {
        Serial.print("Invalid response: ");
        Serial.println(resp);
        error_cb();
        break;  // delay and retry
      }

      // Parse game id from header
      String game_id;
      while (true) {
        String line = client.readStringUntil('\n');
        line.trim();
        line.toLowerCase();
        if (line.startsWith("x-game-id: ")) {
          game_id = line.substring(11);
          game_id.trim();
        }
        if (line == "") {
          break;
        }
      }

      // Read game stat
      String state = client.readStringUntil('\n');
      state.trim();
      Serial.print("State: ");
      Serial.println(state);

      // Exec callbacks
      if (state == "started") {
        started_cb();
      } else if (state == "locked") {
          game_locked = true;
          locked_game_id = game_id;
          Serial.print("Game ID locked: ");
          Serial.println(locked_game_id);
        locked_cb();
      } else if (state == "released") {
        released_cb();
      } else {
        Serial.print("Unexpected state: ");
        Serial.println(state);
        error_cb();
      }

    } else {
      // if you couldn't make a connection:
      Serial.println("connection failed");
      error_cb();
    }

    break;  // always break and wait
  }

  delay(5000);  // Wait 5s before next request
}
