# Copyright (C) 2021-2025, dilemma-vr.games
# Shared under MIT license

"""Controller module in a MVC design"""

import time
import re
import os
from pathlib import Path
import requests
import appdirs
from PyQt5.QtCore import QObject, pyqtSlot, pyqtSignal, QThread
from .model import link
from .utils import open_tray, close_tray

DILEMMA_API = "https://dilemma-vr.games/api/devices/"

headers = {
    "User-Agent": "warden-1.0.0",
}

def get_config_path():
    """Return location of config file"""
    config_dir = appdirs.user_config_dir("dilemmawarden")
    return os.path.join(config_dir, "device_id.txt")

def store_device_id(device_id):
    """Write device id to config file"""
    config_path = get_config_path()

    try:
        # Ensure directory
        config_dir = os.path.dirname(config_path)
        Path(config_dir).mkdir(parents=True, exist_ok=True)

        with open(config_path, "w", encoding="utf-8") as config_file:
            config_file.write(device_id)
    except:
        # Don't break on error
        pass

def load_device_id():
    """Read and return device id from config file or empty string"""
    config_path = get_config_path()
    try:
        with open(config_path, encoding="utf-8") as config_file:
            device_id = config_file.read()
    except FileNotFoundError:
        return ""

    return device_id.strip()

class MainController(QObject):
    """Main controller connecting UI and model"""

    def __init__(self, model):
        super().__init__()

        self._model = model
        self._model.device_id = load_device_id()

    @pyqtSlot(str)
    def change_device_id(self, value):
        """QtSlot called when device_id is changed"""
        self._model.device_id = value

    def change_mode(self, value):
        self._model.mode = value

    @pyqtSlot(int)
    def change_selected_drive(self, index):
        """QtSlot called when drive selected"""
        self._model.selected_drive = index
        if self._model.tray_open:
            open_tray(self.drive_name())
        else:
            close_tray(self.drive_name())

    @pyqtSlot()
    def manual_toggle(self):
        """QtSlot called in manual operation mode to toggle tray"""
        if self._model.tray_open:
            close_tray(self.drive_name())
            self._model.tray_open = False
        else:
            open_tray(self.drive_name())
            self._model.tray_open = True

    @pyqtSlot()
    def remote_toggle(self):
        """QtSlot called in remote mode to toggle active"""
        if self._model.active:
            self._model.link = link.inactive
            self._model.game_id = ""
            self._model.active = False

            # Request thread to exit run()
            self.thread.requestInterruption()
        else:
            self._model.active = True

            store_device_id(self._model.device_id)

            # Create new thread
            self.thread = QThread()
            self.worker = ApiClient()
            self.worker.moveToThread(self.thread)
            self.worker.setUp(self.thread, self._model, self)
            self.thread.start()

    @pyqtSlot()
    def stop_thread(self):
        """Stop background API thread once run() is done"""
        self.thread.quit()
        self.thread.wait()

    @pyqtSlot(str)
    def receive_game_id(self, game_id):
        """QtSlot called when API reports game_id"""
        self._model.game_id = game_id

    @pyqtSlot(int)
    def receive_link_update(self, update):
        """QtSlot called when API connection status changes"""
        self._model.link = update
        if update == link.released:
            if not self._model.tray_open:
                self._model.tray_open = True
                open_tray(self.drive_name())
        else:
            if self._model.tray_open:
                self._model.tray_open = False
                close_tray(self.drive_name())

    def drive_name(self):
        """Return currently selected drive name"""
        idx = self._model.selected_drive
        return self._model.drives[idx] or "cdrom"

class ApiClient(QObject):
    """API client background worker"""
    game_id_changed = pyqtSignal(str)
    link_changed = pyqtSignal(int)
    finished = pyqtSignal()

    def setUp(self, thread, model, main_controller):
        # Sanitize device id
        self.device_id = re.sub("\W+", "", model.device_id)

        # Forward thread control
        thread.started.connect(self.run)

        # Forward changes to controller
        self.game_id_changed.connect(main_controller.receive_game_id)
        self.link_changed.connect(main_controller.receive_link_update)
        self.finished.connect(main_controller.stop_thread)

    def run(self):
        """Background thread main routine"""

        # Mapping from API response to link status
        state_map = {
            "started": link.started,
            "released": link.released,
            "locked": link.locked,
        }

        # Resolve API URL
        api_url = os.environ.get("DILEMMA_API", DILEMMA_API)

        i = -1  # Use to throttle API requests
        N = 100 # Idle cycles (each 0.05s)

        # Store game ID once the game reached 'locked' state
        locked_game_id = None
        
        sess = requests.Session()

        # Exit loop and finish on request
        while not QThread.currentThread().isInterruptionRequested():
            i += 1
            if i % N != 0:
                time.sleep(0.05)
                continue

            try:
                # The API always returns the latest game started by the device
                # owner. To prevent players from starting a new game in case
                # of defeat, DilemmaWarden lockes the API to a specific game
                # ID once the game reaches locked state.
                param = ("?gameid=" + locked_game_id) if locked_game_id else ""
                resp = sess.get(api_url + self.device_id + param,
                                headers=headers)
            except requests.exceptions.ConnectionError as e:
                print(e)
                self.link_changed.emit(link.error)
                continue
                
            if not resp.ok:
                print("Unexpected return code: %d" % resp.status_code)
                self.link_changed.emit(link.error)
                continue

            state = resp.text
            self.link_changed.emit(state_map.get(state, link.error))

            header = "X-Game-ID"
            if header in resp.headers:
                game_id = resp.headers[header]
                self.game_id_changed.emit(game_id)
                print("Game ID changed: %s" % game_id)
                if state == "locked":
                    # Lock DilemmaWarden to specific game ID
                    print("Game ID locked")
                    locked_game_id = game_id

        # Notify controller to stop this thread
        self.finished.emit()
